/*
 * Razer USB keyboard device driver
 *
 *  Copyright (c) 2014 Milan Votava (votava@mageo . cz)
 *  Copyright (c) 2013 Zach Carlson (FxChiP@Gmail . com)
 *
 * Bases on:	usb_skeleton.c v2.2 by Greg Kroah-Hartman
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * Trademarks are the property of their respective owners.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/usb.h>
#include <linux/input/mt.h>
#include <linux/input.h>
#include <linux/usb/input.h>

#define MODULE_NAME "razer_kbd"

static int rzr_debug, rzr_ep_interval = 1;
module_param_named(debug, rzr_debug, int, 0600);
MODULE_PARM_DESC(debug, "Debug level (0=off, 1=on)");

module_param_named(ep_interval, rzr_ep_interval, int, 0600);
MODULE_PARM_DESC(ep_interval, "USB endpoint interval (1-16)");

#define rzr_dprintk(format, a...)                       \
    do {                                \
        if (rzr_debug)                      \
            printk(KERN_DEBUG MODULE_NAME ": " format, ##a);         \
    } while (0)

#define rzr_printk(level, format, a...)                       \
    printk(level MODULE_NAME ": " format, ##a);

#define USB_VENDOR_ID_SYNUSB_RAZER      0x1532

#define USB_DEVICE_ID_SYNUSB_RAZER_BLADE_TP    0x0116  	/* Razer Blade Touchpad */

#define SYNUSB_IO_ALWAYS		(1 << 5)
#define SYNUSB_SHARES_DEVICE            (1 << 7) /* Shares device with other inputs */

#define USB_DEVICE_SYNUSB_RAZER(prod, kind)	        \
	USB_DEVICE(USB_VENDOR_ID_SYNUSB_RAZER,		\
		   USB_DEVICE_ID_SYNUSB_RAZER_##prod),	        \
	.driver_info = (kind),

#define SYNUSB_RECV_SIZE	16

typedef enum {
    RAZER_TRACKPAD   = 0,
    RAZER_KEYBOARD   = 2,
    RAZER_MACROS     = 1,
    RAZER_CONTROLLER = 3,
    RAZER_UNKNOWN    = 4
} razer_input_type;

typedef enum {
    RAZER_MUXED_KEYBOARD = 1,
    RAZER_MUXED_FN       = 2,
    RAZER_MUXED_XTRA     = 4,
    RAZER_MUXED_UNKNOWN  = 0
} razer_mux_input_type;

#define RAZER_IS_MACROKEY 0x50
#define RAZER_FN_KEY 0x01
#define RAZER_RAZER_KEY 0x02
#define RAZER_BRIGHTNESS_UP_KEY 0x06
#define RAZER_BRIGHTNESS_DOWN_KEY 0x07

#define unk KEY_UNKNOWN

#define MODIFIER_CTRL_L     (1 << 0)
#define MODIFIER_SHIFT_L    (1 << 1)    /* KEY_LEFTSHIFT */
#define MODIFIER_ALT_L      (1 << 2)    /* KEY_LEFTALT */
#define MODIFIER_WIN        (1 << 3)    /* KEY_LEFTMETA */
#define MODIFIER_SHIFT_R    (1 << 5)
#define MODIFIER_ALT_R      (1 << 6)

#define FN_MUTE_KEY      226
#define FN_VOLUME_DOWN_KEY      234
#define FN_VOLUME_UP_KEY        233

#define FN_BRIGHTNESS_DOWN_KEY  112
#define FN_BRIGHTNESS_UP_KEY    111
                                

//static const unsigned char hid_keyboard[] = {
#if 0
static const u16 hid_keyboard[] = {
      0,  0,  0,  0, 30, 48, 46, 32, 18, 33, 34, 35, 23, 36, 37, 38,
     50, 49, 24, 25, 16, 19, 31, 20, 22, 47, 17, 45, 21, 44,  2,  3,
      4,  5,  6,  7,  8,  9, 10, 11, 28,  1, 14, 15, 57, 12, 13, 26,
     27, 43, 43, 39, 40, 41, 51, 52, 53, 58, 59, 60, 61, 62, 63, 64,
     65, 66, 67, 68, 87, 88, 99, 70,119,110,102,104,111,107,109,106,
    105,108,103, 69, 98, 55, 74, 78, 96, 79, 80, 81, 75, 76, 77, 71,
     72, 73, 82, 83, 86,127,116,117,183,184,185,186,187,188,189,190,
    191,192,193,194,134,138,130,132,128,129,131,137,133,135,136,113,
    115,114,unk,unk,unk,121,unk, 89, 93,124, 92, 94, 95,unk,unk,unk,
    122,123, 90, 91, 85,unk,unk,unk,unk,unk,unk,unk,111,unk,unk,unk,
    unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,
    unk,unk,unk,unk,unk,unk,179,180,unk,unk,unk,unk,unk,unk,unk,unk,
    unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,
    unk,unk,unk,unk,unk,unk,unk,unk,111,unk,unk,unk,unk,unk,unk,unk,
     29, 42, 56,125, 97, 54,100,126,164,166,165,163,161,115,114,113,
    /*
    150,158,159,128,136,177,178,176,142,152,173,140,unk,unk,unk,unk
    */
    150,158,159,128,136,177,178,176,142,152,173,140,unk,unk,224,225
    ,BTN_0,BTN_1,BTN_2,BTN_3,BTN_4,BTN_5,BTN_6,BTN_7,BTN_8,BTN_9
};

#else
static const u16 hid_keyboard[] = {
    /* 0 - 15 */
     KEY_RESERVED, KEY_RESERVED, KEY_RESERVED, KEY_RESERVED, KEY_A, KEY_B, KEY_C, KEY_D, KEY_E, KEY_F, KEY_G, KEY_H, KEY_I, KEY_J, KEY_K, KEY_L,

    /* 16 - 31 */
     KEY_M, KEY_N, KEY_O, KEY_P, KEY_Q, KEY_R, KEY_S, KEY_T, KEY_U, KEY_V, KEY_W, KEY_X, KEY_Y, KEY_Z, KEY_1, KEY_2,

    /* 32 - 47 */
     KEY_3, KEY_4, KEY_5, KEY_6, KEY_7, KEY_8, KEY_9, KEY_0, KEY_ENTER, KEY_ESC, KEY_BACKSPACE, KEY_TAB, KEY_SPACE, KEY_MINUS, KEY_EQUAL, KEY_LEFTBRACE,

    /* 48 - 63 */
     KEY_RIGHTBRACE, KEY_BACKSLASH, KEY_BACKSLASH, KEY_SEMICOLON, KEY_APOSTROPHE, KEY_GRAVE, KEY_COMMA, KEY_DOT, KEY_SLASH, KEY_CAPSLOCK, KEY_F1, KEY_F2, KEY_F3, KEY_F4, KEY_F5, KEY_F6,

    /* 64 - 79 */
     KEY_F7, KEY_F8, KEY_F9, KEY_F10, KEY_F11, KEY_F12, KEY_SYSRQ, KEY_SCROLLLOCK,KEY_PAUSE,KEY_INSERT,KEY_HOME,KEY_PAGEUP,KEY_DELETE,KEY_END,KEY_PAGEDOWN,KEY_RIGHT,

    /* 80 - 95 */
    KEY_LEFT,KEY_DOWN,KEY_UP, KEY_NUMLOCK, KEY_KPSLASH, KEY_KPASTERISK, KEY_KPMINUS, KEY_KPPLUS, KEY_KPENTER, KEY_KP1, KEY_KP2, KEY_KP3, KEY_KP4, KEY_KP4, KEY_KP6, KEY_KP7,

    /* 96 - 111 */
     KEY_KP8, KEY_KP9, KEY_KP0, KEY_KPDOT, KEY_102ND,KEY_COMPOSE,KEY_POWER,KEY_KPEQUAL,KEY_F13,KEY_F14,KEY_F15,KEY_F16,KEY_F17,KEY_F18,KEY_F19,KEY_F20,

    /* 112 - 127 */
    KEY_F21,KEY_F22,KEY_F23,KEY_F24,KEY_OPEN,KEY_HELP,KEY_PROPS,KEY_FRONT,KEY_STOP,KEY_AGAIN,KEY_UNDO,KEY_CUT,KEY_COPY,KEY_PASTE,KEY_FIND,KEY_MUTE,

    /* 128 - 143 */
    KEY_VOLUMEUP,KEY_VOLUMEDOWN,unk,unk,unk,KEY_KPCOMMA,unk, KEY_RO, KEY_KATAKANAHIRAGANA,KEY_YEN, KEY_HENKAN, KEY_MUHENKAN, KEY_KPJPCOMMA,unk,unk,unk,

    /* 144 - 159 */
    KEY_HANGEUL,KEY_HANJA, KEY_KATAKANA, KEY_HIRAGANA, KEY_ZENKAKUHANKAKU,unk,unk,unk,unk,unk,unk,unk,KEY_DELETE,unk,unk,unk,

    /* 160 - 175 */
    unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,

    /* 176 - 191 */
    unk,unk,unk,unk,unk,unk,KEY_KPLEFTPAREN,KEY_KPRIGHTPAREN,unk,unk,unk,unk,unk,unk,unk,unk,
    /* 192 - 207 */
    unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,unk,
    /* 208 - 223 */
    unk,unk,unk,unk,unk,unk,unk,unk,KEY_DELETE,unk,unk,unk,unk,unk,unk,unk,
    /* 224 - 239 */
     KEY_LEFTCTRL, KEY_LEFTSHIFT, KEY_LEFTALT,KEY_LEFTMETA, KEY_RIGHTCTRL,KEY_RIGHTSHIFT,KEY_RIGHTALT,KEY_RIGHTMETA,KEY_PLAYPAUSE,KEY_STOPCD,KEY_PREVIOUSSONG,KEY_NEXTSONG,KEY_EJECTCD,KEY_VOLUMEUP,KEY_VOLUMEDOWN,KEY_MUTE,
    /* 240 - 255 */
    KEY_WWW,KEY_BACK,KEY_FORWARD,KEY_STOP,KEY_FIND,KEY_SCROLLUP,KEY_SCROLLDOWN,KEY_EDIT,KEY_SLEEP,KEY_COFFEE,KEY_REFRESH,KEY_CALC,unk,KEY_SWITCHVIDEOMODE,KEY_BRIGHTNESSDOWN,KEY_BRIGHTNESSUP
    /* 256 - 271 */
    ,KEY_F13,KEY_F14,KEY_F15,KEY_F16,KEY_F17,KEY_F18,KEY_F19,KEY_F20,KEY_F21,KEY_F22, KEY_SWITCHVIDEOMODE
};
#endif

struct rzr_kbd {
	struct usb_device *udev;
	struct usb_interface *intf;
	struct urb *urb_kbd;
	dma_addr_t *data;

	/* input device related data structures */
	struct input_dev *input;
	char name[128];
	char phys[64];

	/* characteristics of the device */
	unsigned long flags;

    u8 modifiers_state;
    u8 key_state[15];
    u8 xkey_state[15];
    u8 fnkey_state;

    //u8 keycode[ARRAY_SIZE(hid_keyboard)];
    u16 keycode[ARRAY_SIZE(hid_keyboard)];


    /* LED */
    dma_addr_t *leds_dma;
	struct urb *urb_led;
    unsigned char newleds;
    unsigned char *new;
    struct usb_ctrlrequest *cr;
    unsigned char *leds;
         
    spinlock_t leds_lock;
    bool led_urb_submitted;
};

static int rzr_kbd_xkey_to_keycode(u8 xkey)
{
    if (xkey & RAZER_IS_MACROKEY) {
        return KEY_F13 + (xkey & 0x0f);
    } else {
        int key;
        switch(xkey) {
            case RAZER_FN_KEY:
                key = KEY_FN; 
                break;
            case RAZER_RAZER_KEY:
                key = KEY_VENDOR; 
                break;
            case 5:
                key = KEY_SWITCHVIDEOMODE; 
                break;
            case RAZER_BRIGHTNESS_DOWN_KEY:
                key = KEY_BRIGHTNESSDOWN; 
                break;
            case RAZER_BRIGHTNESS_UP_KEY:
                key = KEY_BRIGHTNESSUP; 
                break;
            default:
                printk(KERN_WARNING "unknown xkey scancode %d\n", xkey);
                key = KEY_UNKNOWN; 
        }
        return key;
    }
}

static int rzr_kbd_fnkey_to_keycode(u8 fnkey)
{
    int key;
    switch(fnkey) {
        case FN_MUTE_KEY:
            key = KEY_MUTE;
            break;
        case FN_VOLUME_DOWN_KEY:
            key = KEY_VOLUMEDOWN; 
            break;
        case FN_VOLUME_UP_KEY:
            key = KEY_VOLUMEUP; 
            break;
        case FN_BRIGHTNESS_DOWN_KEY:
            key = KEY_BRIGHTNESSDOWN; 
            break;
        case FN_BRIGHTNESS_UP_KEY:
            key = KEY_BRIGHTNESSUP; 
            break;
        case 181:
            key = KEY_NEXTSONG; 
            break;
        case 182:
            key = KEY_PREVIOUSSONG; 
            break;
        case 205:
            key = KEY_PLAYPAUSE; 
            break;
        default:
            printk(KERN_WARNING "unknown fnkey scancode %d\n", fnkey);
            key = KEY_UNKNOWN; 
    }
    return key;
}

static void rzr_kbd_input_event(struct rzr_kbd *kbd, unsigned int type, unsigned int code, int value)
{
    if (rzr_debug) {
        /*
        printk (KERN_INFO "iev %d %d %d\n", type, code,  value);
        */
        rzr_dprintk("iev %d %d %d\n", type, code,  value);
    }

    input_event(kbd->input, type, code,  value);
}

static u8 *rzr_kbd_find_key(u8 *p, u8 key, int n)
{
    int i;

    for (i = 0; i < n && p[i]; i++) {
        if (p[i] == key)
            return p + i;
    }
    return NULL;
}

static void rzr_kbd_key_modifier(struct rzr_kbd *kbd, u8 modifiers_state, u8 modifiers, u8 modifier, int modifier_key)
{
    int modifier_state = modifiers & modifier;

    if ((modifiers_state & modifier) != modifier_state) {
        rzr_kbd_input_event(kbd, EV_KEY, modifier_key, !!modifier_state);
    }
}

static void rzr_kbd_process_key_modifiers(struct rzr_kbd *kbd, u8 modifiers)
{
    u8 m = kbd->modifiers_state;

    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_CTRL_L, KEY_LEFTCTRL);
    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_SHIFT_L, KEY_LEFTSHIFT);
    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_SHIFT_R, KEY_RIGHTSHIFT);
    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_WIN, KEY_LEFTMETA);
    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_ALT_L, KEY_LEFTALT);
    rzr_kbd_key_modifier(kbd, m, modifiers, MODIFIER_ALT_R, KEY_RIGHTALT);

    kbd->modifiers_state = modifiers;
}

static void rzr_kbd_process_fnkey(struct rzr_kbd *kbd, u8 fnkey)
{
    u8 s = kbd->fnkey_state;

    if (s && s != fnkey) {
        rzr_kbd_input_event(kbd, EV_KEY, rzr_kbd_fnkey_to_keycode(s), 0);
    }

    if (fnkey) {
        rzr_kbd_input_event(kbd, EV_KEY, rzr_kbd_fnkey_to_keycode(fnkey), 1);
    }

    kbd->fnkey_state = fnkey;
}

static void rzr_kbd_update_keys_state(u8 *old, u8 *new, u8* up, u8 *down, int new_offset)
{
    int i;
    u8 *po = old, *pu = up, *pd = down;

    /* store released keys in up */
    for (i = 0; i < 15 && old[i]; i++) {
        u8 c = old[i];             
        
        if (!rzr_kbd_find_key(new + new_offset, c, 15 - new_offset)) {
            *(pu++) = c;            
        }
    }
    /* end marker */
    if (pu < up + 15)
        *pu = 0;

    /* new offset: keys in comparison with xkeys has 1st keys always 0 */
    /* store newly pressed keys in down */
    for (i = new_offset; i < 15 && new[i]; i++) {
        u8 c = new[i];             
        
        if (!rzr_kbd_find_key(old, c, 15)) {
            *(pd++) = c;            
        }
        /* and refresh old... */
        *(po++) = c;
    }
    /* end markers */
    if (pd < down + 15)
        *pd = 0;

    if (po < old + 15)
        *po = 0;
}

static void rzr_kbd_raw_event(struct rzr_kbd *kbd)
{
    int i;
    u8 down[15], up[15]; 
    u8 *d = (u8*)kbd->data;
    int size = 16;

    /*
    printk(KERN_INFO "rzr_kbd_raw_event %d\n", kbd->intf->cur_altsetting->desc.bInterfaceNumber);

    for (i = 0; i < size; i++) {
        printk(KERN_INFO " %d", d[i]);
    }
    printk(KERN_INFO "\n");
    */

    if (rzr_debug) {
        /*
        printk (KERN_INFO "rzr_kbd_raw_event %d: %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n", kbd->intf->cur_altsetting->desc.bInterfaceNumber,
        */
        rzr_dprintk("rzr_kbd_raw_event %d: %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n", kbd->intf->cur_altsetting->desc.bInterfaceNumber,
        d[0], d[1], d[2], d[3], d[4], d[5], d[6], d[7],
        d[8], d[9], d[10], d[11], d[12], d[13], d[14], d[15]
        );
    }

    if (size) {
        switch (d[0]) {
            case RAZER_MUXED_KEYBOARD:
                /*
                hid_info(hdev, "Razer Keyboard %x:\n", d[1]);
                for (i = 2; i < size; i++) {
                    if (d[i]) {
                        u8 key = hid_keyboard[d[i]];
                        hid_info(hdev, "  Keycode entered: %d\n", key);
                        //input_event(kbd->input, EV_KEY, key, 1);
                    }
                }
                */

                rzr_kbd_process_key_modifiers(kbd, d[1]);

                /* if more then 15 keys pressed, all keys are 1 */
                if (d[2] != 1) {

                    rzr_kbd_update_keys_state(kbd->key_state, d + 1, up, down, 1);
                    for (i = 0; i < 15 && up[i]; i++) {
                        /*hid_info(hdev, "up: %d\n", up[i]);*/
                        rzr_kbd_input_event(kbd, EV_KEY, hid_keyboard[up[i]], 0);
                    }
                    for (i = 0; i < 15 && down[i]; i++) {
                        /*hid_info(hdev, "down: %d\n", down[i]);*/
                        rzr_kbd_input_event(kbd, EV_KEY, hid_keyboard[down[i]], 1);
                    }
                }

                input_sync(kbd->input);

                break;
            case RAZER_MUXED_FN:
                rzr_kbd_process_fnkey(kbd, d[1]);

                input_sync(kbd->input);

                break;
            case RAZER_MUXED_XTRA:
                /*
                hid_info(hdev, "Razer Extra Buttons:\n");
                for (i = 1; i < size; i++) {
                    u8 xkey = d[i];
                    if (xkey) {
                        if (xkey & RAZER_IS_MACROKEY) {
                            hid_info(hdev, "  Macro key: %d\n", xkey & 0x0F);
                        } else {
                            char *key_name = NULL;
                            switch(d[i]) {
                                case RAZER_FN_KEY:
                                    key_name = "FN";
                                    break;
                                case RAZER_RAZER_KEY:
                                    key_name = "RAZER";
                                    break;
                                case RAZER_BRIGHTNESS_DOWN_KEY:
                                    key_name = "BRIGHTNESS_DOWN";
                                    break;
                                case RAZER_BRIGHTNESS_UP_KEY:
                                    key_name = "BRIGHTNESS_UP";
                                    break;
                                default:
                                    key_name = "UNKNOWN";
                            }
                            hid_info(hdev, "  Extra key: %s\n", key_name);
                        }
                    }
                }
                */

                rzr_kbd_update_keys_state(kbd->xkey_state, d + 1, up, down, 0);

                for (i = 0; i < 15 && up[i]; i++) {
                    rzr_kbd_input_event(kbd, EV_KEY, rzr_kbd_xkey_to_keycode(up[i]), 0);
                }
                for (i = 0; i < 15 && down[i]; i++) {
                    rzr_kbd_input_event(kbd, EV_KEY, rzr_kbd_xkey_to_keycode(down[i]), 1);
                }

                input_sync(kbd->input);

                break;
            default:
                break;
        }

    }
}

static void rzr_kbd_irq(struct urb *urb)
{
	struct rzr_kbd *kbd = urb->context;
	int error;

    /*
    printk (KERN_INFO "razer_irq\n");
    */
	/* Check our status in case we need to bail out early. */
	switch (urb->status) {
	case 0:
		usb_mark_last_busy(kbd->udev);
		break;

	/* Device went away so don't keep trying to read from it. */
	case -ECONNRESET:
	case -ENOENT:
	case -ESHUTDOWN:
		return;

	default:
		goto resubmit;
		break;
	}

    rzr_kbd_raw_event(kbd);

resubmit:
	error = usb_submit_urb(urb, GFP_ATOMIC);
	if (error && error != -EPERM)
		rzr_printk(KERN_ERR,
            "%s - usb_submit_urb failed with result: %d",
			__func__, error);
}

static void rzr_kbd_kbd_led(struct urb *urb)
{
        unsigned long flags;
        struct rzr_kbd *kbd = urb->context;

        if (urb->status)
                rzr_printk(KERN_WARNING, "led urb status %d received\n",
                         urb->status);
 
        spin_lock_irqsave(&kbd->leds_lock, flags);

        if (*(kbd->leds) == kbd->newleds){
                kbd->led_urb_submitted = false;
                spin_unlock_irqrestore(&kbd->leds_lock, flags);
                return;
        }
 
        *(kbd->leds) = kbd->newleds;
        
        kbd->urb_led->dev = kbd->udev;
        if (usb_submit_urb(kbd->urb_led, GFP_ATOMIC)){
                rzr_printk(KERN_ERR, "usb_submit_urb(leds) failed\n");
                kbd->led_urb_submitted = false;
        }
        spin_unlock_irqrestore(&kbd->leds_lock, flags);
         
}

static int rzr_kbd_kbd_event(struct input_dev *dev, unsigned int type,
                          unsigned int code, int value)
{
    unsigned long flags;
    struct rzr_kbd *kbd = input_get_drvdata(dev);
 
    rzr_dprintk ("razer_kbd_event\n");

    if (type != EV_LED)
        return -1;
 
    spin_lock_irqsave(&kbd->leds_lock, flags);
    kbd->newleds = (!!test_bit(LED_KANA,    dev->led) << 3) | (!!test_bit(LED_COMPOSE, dev->led) << 3) |
                   (!!test_bit(LED_SCROLLL, dev->led) << 2) | (!!test_bit(LED_CAPSL,   dev->led) << 1) |
                   (!!test_bit(LED_NUML,    dev->led));

    if (kbd->led_urb_submitted){
            spin_unlock_irqrestore(&kbd->leds_lock, flags);
            return 0;
    }

    if (*(kbd->leds) == kbd->newleds){
           spin_unlock_irqrestore(&kbd->leds_lock, flags);
           return 0;
    }

    *(kbd->leds) = kbd->newleds;
     
    kbd->urb_led->dev = kbd->udev;
    if (usb_submit_urb(kbd->urb_led, GFP_ATOMIC))
            pr_err("usb_submit_urb(leds) failed\n");
    else
            kbd->led_urb_submitted = true;
     
    spin_unlock_irqrestore(&kbd->leds_lock, flags);
    return 0;
}

static struct usb_endpoint_descriptor *
rzr_kbd_get_in_endpoint(struct usb_host_interface *iface)
{

	struct usb_endpoint_descriptor *endpoint;
	int i;

	for (i = 0; i < iface->desc.bNumEndpoints; ++i) {
		endpoint = &iface->endpoint[i].desc;

		if (usb_endpoint_is_int_in(endpoint)) {
			/* we found our interrupt in endpoint */
			return endpoint;
		}
	}

	return NULL;
}

static int rzr_kbd_open(struct input_dev *dev)
{
	struct rzr_kbd *kbd = input_get_drvdata(dev);
	int retval;

	retval = usb_autopm_get_interface(kbd->intf);
	if (retval) {
		rzr_printk(KERN_ERR,
			"%s - usb_autopm_get_interface failed, error: %d\n",
			__func__, retval);
		return retval;
	}

	retval = usb_submit_urb(kbd->urb_kbd, GFP_KERNEL);
	if (retval) {
		rzr_printk(KERN_ERR,
			"%s - usb_submit_urb failed, error: %d\n",
			__func__, retval);
		retval = -EIO;
		goto out;
	}

	kbd->intf->needs_remote_wakeup = 1;

out:
	usb_autopm_put_interface(kbd->intf);
	return retval;
}

static void rzr_kbd_close(struct input_dev *dev)
{
	struct rzr_kbd *kbd = input_get_drvdata(dev);
	int autopm_error;

	autopm_error = usb_autopm_get_interface(kbd->intf);

	usb_kill_urb(kbd->urb_kbd);
	kbd->intf->needs_remote_wakeup = 0;

	if (!autopm_error)
		usb_autopm_put_interface(kbd->intf);
}

static void rzr_hack_set_ep_interval(struct usb_device *udev, int intf_n, int alt_n, int ep_n, int interval)
{
    struct usb_host_interface *alt;

    alt = &udev->actconfig->intf_cache[intf_n]->altsetting[alt_n];
    alt->endpoint[ep_n].desc.bInterval = interval;
}

static int usb_kbd_alloc_mem(struct usb_device *dev, struct rzr_kbd *kbd)
{
    if (!(kbd->urb_kbd = usb_alloc_urb(0, GFP_KERNEL)))
        return -1;
    if (!(kbd->urb_led = usb_alloc_urb(0, GFP_KERNEL)))
        return -1;
    if (!(kbd->data = usb_alloc_coherent(dev, SYNUSB_RECV_SIZE, GFP_ATOMIC, &kbd->urb_kbd->transfer_dma)))
        return -1;
    if (!(kbd->cr = kmalloc(sizeof(struct usb_ctrlrequest), GFP_KERNEL)))
        return -1;
    if (!(kbd->leds = usb_alloc_coherent(dev, 1, GFP_ATOMIC, &kbd->urb_led->transfer_dma)))
        return -1;

    return 0;
}

static void usb_kbd_free_mem(struct usb_device *dev, struct rzr_kbd *kbd)
{
    usb_free_coherent(dev, SYNUSB_RECV_SIZE, kbd->data, kbd->urb_kbd->transfer_dma);
    kfree(kbd->cr);
    usb_free_coherent(dev, 1, kbd->leds, kbd->urb_led->transfer_dma);

    usb_free_urb(kbd->urb_kbd);
    usb_free_urb(kbd->urb_led);
}

static int rzr_kbd_probe(struct usb_interface *intf,
			const struct usb_device_id *id)
{
	struct usb_device *udev = interface_to_usbdev(intf);
	struct usb_endpoint_descriptor *ep;
	struct rzr_kbd *kbd;
	struct input_dev *input_dev;
	unsigned int intf_num = intf->cur_altsetting->desc.bInterfaceNumber;
    /*
	unsigned int altsetting = min(intf->num_altsetting, 1U);
    */
	unsigned int altsetting = min(intf->num_altsetting, 0U);
	int error;
	size_t recv_size = SYNUSB_RECV_SIZE;
    int i;
    /*
    struct usb_host_interface *alt;
    */

    rzr_dprintk ("razer_kbd_probe %s %s %s %d %d\n", udev->devpath, udev->dev.driver->name, udev->dev.driver->owner->name, udev->state, intf->condition);

	if (id->driver_info & SYNUSB_SHARES_DEVICE) {
		/* XXX: Trackpad on interface 0. Hack to ignore other interfaces until better method found. */
		/* Must NOT bind to keyboards, etc. */
        /*
		if (intf_num < 1 || intf_num > 2)
			return -ENODEV;
        */
        if (intf->cur_altsetting->desc.bInterfaceProtocol != 1) /* KEYBOARD */
            return -ENODEV;
	}

    if (rzr_ep_interval) {
        rzr_ep_interval = clamp(rzr_ep_interval, 1, 16);

        /* example: http://www.spinics.net/lists/linux-usb/msg95088.html

        alt = &gspca_dev->dev->actconfig->intf_cache[0]->altsetting[1];
        alt->endpoint[0].desc.wMaxPacketSize = cpu_to_le16(packet_size);
        */

        /*
        alt = &udev->actconfig->intf_cache[0]->altsetting[0];
        alt->endpoint[0].desc.bInterval = rzr_ep_interval;
        */

        /*
        rzr_hack_set_ep_interval(udev, 0, 0, 0, rzr_ep_interval);
        */
        rzr_hack_set_ep_interval(udev, intf_num, altsetting, 0, rzr_ep_interval);

        error = usb_set_interface(udev, intf_num, altsetting);
        if (error) {
            rzr_printk(KERN_ERR,
                "Can not set alternate setting to %i, error: %i",
                altsetting, error);
            return error;
        }

        intf->cur_altsetting->endpoint[0].desc.bInterval = rzr_ep_interval;
    }

	ep = rzr_kbd_get_in_endpoint(intf->cur_altsetting);
	if (!ep)
		return -ENODEV;

	kbd = kzalloc(sizeof(*kbd), GFP_KERNEL);
	input_dev = input_allocate_device();
	if (!kbd || !input_dev) {
		error = -ENOMEM;
		goto fail1;
	}

    if (usb_kbd_alloc_mem(udev, kbd))
        goto fail2;

	kbd->udev = udev;
	kbd->intf = intf;
	kbd->input = input_dev;


	kbd->flags = id->driver_info;

	usb_fill_int_urb(kbd->urb_kbd, udev,
			 usb_rcvintpipe(udev, ep->bEndpointAddress),
			 kbd->data, recv_size,
			 rzr_kbd_irq, kbd,
             rzr_ep_interval ? rzr_ep_interval : ep->bInterval);

	kbd->urb_kbd->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;


	if (udev->manufacturer)
		strlcpy(kbd->name, udev->manufacturer,
			sizeof(kbd->name));

	if (udev->product) {
		if (udev->manufacturer)
			strlcat(kbd->name, " ", sizeof(kbd->name));
		strlcat(kbd->name, udev->product, sizeof(kbd->name));
	}

	if (!strlen(kbd->name))
		snprintf(kbd->name, sizeof(kbd->name),
			 "USB Razer Device %04x:%04x",
			 le16_to_cpu(udev->descriptor.idVendor),
			 le16_to_cpu(udev->descriptor.idProduct));

    strlcat(kbd->name, " (Keyboard)", sizeof(kbd->name));

	if (kbd->flags & SYNUSB_SHARES_DEVICE) 
		strlcat(kbd->name, " (shared)", sizeof(kbd->name));


	usb_make_path(udev, kbd->phys, sizeof(kbd->phys));
	strlcat(kbd->phys, "/input0", sizeof(kbd->phys));

	input_set_drvdata(input_dev, kbd);

    memcpy(kbd->keycode, hid_keyboard, sizeof(kbd->keycode));

	input_dev->name = kbd->name;
	input_dev->phys = kbd->phys;
	usb_to_input_id(udev, &input_dev->id);
	input_dev->dev.parent = &kbd->intf->dev;

    input_dev->keycode = kbd->keycode;
    //input_dev->keycodesize = sizeof(unsigned char);
    input_dev->keycodesize = sizeof(u16);
    input_dev->keycodemax = ARRAY_SIZE(hid_keyboard);

    /*for (i = 0; i < 255; i++)*/
    /*for (i = 0; i < 256; i++)*/
    for (i = 0; i < ARRAY_SIZE(hid_keyboard); i++)
        set_bit(kbd->keycode[i], input_dev->keybit);

    clear_bit(0, input_dev->keybit); 

	if (!(kbd->flags & SYNUSB_IO_ALWAYS)) {
		input_dev->open = rzr_kbd_open;
		input_dev->close = rzr_kbd_close;
	}

	__set_bit(EV_KEY, input_dev->evbit);
	__set_bit(EV_REP, input_dev->evbit);

    /* LED */
    spin_lock_init(&kbd->leds_lock);

    kbd->cr->bRequestType = USB_TYPE_CLASS | USB_RECIP_INTERFACE;
    kbd->cr->bRequest = 0x09;
    kbd->cr->wValue = cpu_to_le16(0x200);
    kbd->cr->wIndex = cpu_to_le16(intf->cur_altsetting->desc.bInterfaceNumber);
    kbd->cr->wLength = cpu_to_le16(1);

    usb_fill_control_urb(kbd->urb_led, udev, usb_sndctrlpipe(udev, 0),
              (void *) kbd->cr, kbd->leds, 1,
              rzr_kbd_kbd_led, kbd);
    kbd->urb_led->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

    input_dev->event = rzr_kbd_kbd_event;
	__set_bit(EV_LED, input_dev->evbit);
    set_bit(LED_CAPSL, input_dev->ledbit);


	usb_set_intfdata(intf, kbd);

	if (kbd->flags & SYNUSB_IO_ALWAYS) {
		error = rzr_kbd_open(input_dev);
		if (error)
			goto fail2;
	}

	error = input_register_device(input_dev);
	if (error) {
		rzr_printk(KERN_ERR,
			"Failed to register input device, error %d\n",
			error);
		goto fail3;
	}

	return 0;

fail3:
	if (kbd->flags & SYNUSB_IO_ALWAYS)
		rzr_kbd_close(kbd->input);
fail2:
    usb_kbd_free_mem(udev, kbd);

fail1:
	input_free_device(input_dev);
	kfree(kbd);
	usb_set_intfdata(intf, NULL);

	return error;
}

static void rzr_kbd_disconnect(struct usb_interface *intf)
{
	struct rzr_kbd *kbd = usb_get_intfdata(intf);

    rzr_dprintk ("rzr_kbd_disconnect\n");

	usb_set_intfdata(intf, NULL);
    if (kbd) {
	    if (kbd->flags & SYNUSB_IO_ALWAYS)
	    	rzr_kbd_close(kbd->input);

        usb_kill_urb(kbd->urb_kbd);

	    input_unregister_device(kbd->input);

        usb_kill_urb(kbd->urb_led);
    
        usb_kbd_free_mem(interface_to_usbdev(intf), kbd);

    	kfree(kbd);
    }
}

static int rzr_kbd_suspend(struct usb_interface *intf, pm_message_t message)
{
	struct rzr_kbd *kbd = usb_get_intfdata(intf);
	struct input_dev *input_dev = kbd->input;

    rzr_dprintk ("rzr_kbd_suspend\n");

	mutex_lock(&input_dev->mutex);
	usb_kill_urb(kbd->urb_kbd);
	usb_kill_urb(kbd->urb_led);
	mutex_unlock(&input_dev->mutex);

	return 0;
}

static int rzr_kbd_resume(struct usb_interface *intf)
{
	struct rzr_kbd *kbd = usb_get_intfdata(intf);
	struct input_dev *input_dev = kbd->input;
	int retval = 0;

    rzr_dprintk ("rzr_kbd_resume\n");

	mutex_lock(&input_dev->mutex);

    /*
	if ((input_dev->users || (kbd->flags & SYNUSB_IO_ALWAYS)) &&
	    usb_submit_urb(kbd->urb_kbd, GFP_NOIO) < 0) {
		retval = -EIO;
	}
    */
	if (input_dev->users || (kbd->flags & SYNUSB_IO_ALWAYS)) {
	    if (usb_submit_urb(kbd->urb_kbd, GFP_NOIO) < 0) {
            retval = -EIO;
        } else if (usb_submit_urb(kbd->urb_led, GFP_NOIO) < 0) {
            retval = -EIO;
        }
	}

	mutex_unlock(&input_dev->mutex);

	return retval;
}

static int rzr_kbd_pre_reset(struct usb_interface *intf)
{
	struct rzr_kbd *kbd = usb_get_intfdata(intf);
	struct input_dev *input_dev = kbd->input;

    rzr_dprintk ("rzr_kbd_pre_reset\n");

	mutex_lock(&input_dev->mutex);
	usb_kill_urb(kbd->urb_kbd);
	usb_kill_urb(kbd->urb_led);

	return 0;
}

static int rzr_kbd_post_reset(struct usb_interface *intf)
{
	struct rzr_kbd *kbd = usb_get_intfdata(intf);
	struct input_dev *input_dev = kbd->input;
	int retval = 0;

    rzr_dprintk ("rzr_kbd_post_reset\n");

    /*
	if ((input_dev->users || (kbd->flags & SYNUSB_IO_ALWAYS)) &&
	    usb_submit_urb(kbd->urb_kbd, GFP_NOIO) < 0) {
		retval = -EIO;
	}
    */
	if (input_dev->users || (kbd->flags & SYNUSB_IO_ALWAYS)) {
	    if (usb_submit_urb(kbd->urb_kbd, GFP_NOIO) < 0) {
            retval = -EIO;
        } else if (usb_submit_urb(kbd->urb_led, GFP_NOIO) < 0) {
            retval = -EIO;
        }
	}

	mutex_unlock(&input_dev->mutex);

	return retval;
}

static int rzr_kbd_reset_resume(struct usb_interface *intf)
{
    rzr_dprintk ("rzr_kbd_reset_resume\n");

	return rzr_kbd_resume(intf);
}

static struct usb_device_id rzr_kbd_idtable[] = {
	{ USB_DEVICE_SYNUSB_RAZER(BLADE_TP, SYNUSB_SHARES_DEVICE) },
	{ }
};
MODULE_DEVICE_TABLE(usb, rzr_kbd_idtable);

static struct usb_driver rzr_kbd_driver = {
	.name		= MODULE_NAME,
	.probe		= rzr_kbd_probe,
	.disconnect	= rzr_kbd_disconnect,
	.id_table	= rzr_kbd_idtable,
	.suspend	= rzr_kbd_suspend,
	.resume		= rzr_kbd_resume,
	.pre_reset	= rzr_kbd_pre_reset,
	.post_reset	= rzr_kbd_post_reset,
	.reset_resume	= rzr_kbd_reset_resume,
	.supports_autosuspend = 1,
};

module_usb_driver(rzr_kbd_driver);

MODULE_AUTHOR("Milan Votava <votava@mageo.cz>");
MODULE_DESCRIPTION("Razer keyboard USB device driver");
MODULE_LICENSE("GPL");
